use strict;
use warnings;

use IO::File;
use XML::Writer;

my $output = new IO::File(">output.xml");

my $writer = new XML::Writer(OUTPUT => $output);
$writer->xmlDecl();

my @operators =
(
["unary plus",1,"+","T operator +(const T& a);"],
["unary minus",1,"-","T operator -(const T& a);"],
["addition",2,"+","T operator +(const T& a, const T& b);"],
["multiplication",2,"*","T operator *(const T &a, const T& b);"],
["subtraction",2,"-","T operator -(const T& a, const T& b);"],
["division",2,"/","T operator /(const T& a, const T& b);"],
["less than",2,"<","bool operator <(const T& a, const T& b);"],
["bitwise not",1,"~","T operator ~(const T& a);"],
["modulo",2,"%","T operator %(const T& a, const T& b);"],
["bitwise and",2,"&","T operator &(const T& a, const T& b);"],
["bitwise or",2,"|","T operator |(const T& a, const T& b);"],
["bitwise xor",2,"^","T operator ^(const T& a, const T& b);"],
["bitwise left shift",2,"<<","T operator <<(const T& a, const T& b);"],
["bitwise right shift",2,">>","T operator >>(const T& a, const T& b);"]
);

my @operands = (

["s1","s1min", "-(2**7)"],
["s1","s1(-1)", "-1"], 
["s1", "s1(0)", "0"],
["s1", "s1(1)", "1"],
["s1","s1max", "(2**7)-1"],
["u1","u1(0)", "0"],
["u1","u1(1)", "1"],
["u1","u1max", "(2**8)-1"],
["s2","s2min", "-(2**15)"],
["s2","s2(-1)", "-1"],
["s2","s2(0)", "0"],
["s2","s2(1)", "1"],
["s2","s2max", "(2**15)-1"],
["u2","u2(0)", "0"],
["u2","u2(1)", "1"],
["u2","u2max", "(2**16)-1"],
["s4","s4(-1)", "-1"],
["s4","s4(0)", "0"],
["s4","s4(1)", "1"],
["s4","s4min", "-(2**31)"],
["s4","s4max", "(2**31)-1"],
["u4","u4(0)", "0"],
["u4","u4(1)", "1"],
["u4","u4max", "(2**32)-1"],
["s8","s8(0)", "0"],
["s8","s8(1)", "1"],
["s8","s8(-1)", "-1"],
["s8","s8min", "-(2**63)"],
["s8","s8max", "(2**63)-1"],
["u8","u8(0)", "0"],
["u8","u8(1)", "1"],
["u8","u8max", "(2**64)-1"]
);

sub trysystem
{
    my $result = {};

    my $ret = (system($_[0].' 1>tmpfile.stdout 2>tmpfile.stderr') == 0);

    $result->{"stdout"} = `cat tmpfile.stdout`;
    $result->{"stderr"} = `cat tmpfile.stderr`;

    if ($ret)
    {
	$result->{"exitcode"} = "ok";
    }
    elsif ($? == -1) 
    {
        $result->{"exitcode"} = "fail $!";
    }
    elsif ($? & 127)
    {
        $result->{"exitcode"} = "fail with signal ". ($? & 127);
    }
    else {
        $result->{"exitcode"} = "fail with exit value " .($? >> 8);
    }

    return $result;
}

sub ignorewhitespace
{
	my $x = $_[0];
	$x =~ s/\s+/ /g;
	$x =~ s/^ //;
	$x =~ s/ $//;
	return $x;
}

sub pyeval
{
	my $expr = $_[0];
	if ($expr =~ /<<|>>/)
	{
		return "skipped";
	}
	my $result = trysystem("echo 'print ($expr)' | python -");
	return ignorewhitespace($result->{"stdout"});
}

sub gcc
{
        my $result = {};

	my $code = "#include \"eval.h\"\n$_[0]";

	open(my $source, ">source.cpp") or die $!;

	print $source $code;

	close($source);

	my $compileok = 0;
	my $ncompiles = 0;
	do
	{
        	my $ret = trysystem("g++ -Wall source.cpp");

		$ncompiles++;

		$result->{"compileoutput"} = "";

		if (defined $ret->{"stderr"})
		{
			$result->{"compileoutput"} .= $ret->{"stderr"} . " ";
		}
		elsif (defined $ret->{"stdout"})
		{
			$result->{"compileoutput"} .= $ret->{"stdout"};
		}

		if ($result->{"compileoutput"} !~ /\S/)
		{
			delete $result->{"compileoutput"};
		}
		else
		{
			$result->{"compileoutput"} = ignorewhitespace($result->{"compileoutput"});
		}

		if ($ret->{"exitcode"} eq 'ok')
        	{
			delete $result->{"status"};

			$compileok = 1;
		}
		else
		{
			$result->{"status"} = "could not compile: ".($ret->{"exitcode"});
			$compileok = 0;
			sleep 3;
		}

	} while ($ncompiles < 3 and !$compileok);

	if (!$compileok)
	{
		return $result;
	}

	my $ret = trysystem("./a.out");

	if ($ret->{"exitcode"} eq "ok")
	{
		$result->{"status"} = "ok";
		$result->{"output"} = ignorewhitespace($ret->{"stdout"});
	}
	else
	{
		$result->{"status"} = $ret->{"exitcode"};
	}

	return $result;
}

$writer->startTag("operators");

foreach my $operator (@operators)
{
    my $name = $operator->[0];
    my $cardinality = $operator->[1];
    my $symbol = $operator->[2];
    my $signature = $operator->[3];

	$writer->startTag('operator', 'name' => $name, 'symbol' => $symbol, 'signature' => $signature);
    if ($cardinality == 1)
    {
	$writer->startTag('unary');
        foreach my $operandinfo (@operands)
        {
		my $operandtype = $operandinfo->[0];
		my $operandvalue = $operandinfo->[1];
		my $operandpyval = $operandinfo->[2];

		$writer->startTag('operand', 'name' => $operandvalue);
            my $expr = "EVAL1($operandtype,$operandvalue,$symbol)";
	   my $pyres = pyeval("$symbol($operandpyval)");
            my $result = gcc($expr);
            if ($result->{"status"} eq 'ok')
            {
		my ($type, $value) = @{eval('['.$result->{"output"}.']')};
		
                $writer->startTag('output', 'type' => $type, 'value' => $value, 'pyval' => $pyres);
		$writer->endTag('output');
            }

            else
            {
		$writer->startTag('status');
                $writer->characters($result->{"status"});
		$writer->endTag('status');
            }
            if (defined $result->{"compileoutput"})
            {
                $writer->startTag('compileoutput');
		$writer->characters($result->{'compileoutput'});
		$writer->endTag('compileoutput');
            }
		$writer->endTag('operand');
        }
	$writer->endTag("unary");
    }
    elsif ($cardinality == 2)
    {
	$writer->startTag('binary');
        foreach my $operand1info (@operands)
        {
		my $operand1type = $operand1info->[0];
		my $operand1value = $operand1info->[1];
		my $operand1pyval = $operand1info->[2];

		$writer->startTag('operand1', 'name' => $operand1value);

            foreach my $operand2info (@operands)
            {
		my $operand2type = $operand2info->[0];
		my $operand2value = $operand2info->[1];
		my $operand2pyval = $operand2info->[2];

		$writer->startTag('operand2', 'name' => $operand2value);
		
                my $expr = "EVAL2($operand1type,$operand1value,$symbol,$operand2type,$operand2value)";
		my $pyres = pyeval("($operand1pyval) $symbol ($operand2pyval)");

                my $result = gcc($expr);
            if ($result->{"status"} eq 'ok')
            {
		my ($type, $value) = @{eval('['.$result->{"output"}.']')};
		
                $writer->startTag('output', 'type' => $type, 'value' => $value, 'pyval' => $pyres);
		$writer->endTag('output');
            }

            else
            {
		$writer->startTag('status');
                $writer->characters($result->{"status"});
		$writer->endTag('status');
            }
            if (defined $result->{"compileoutput"})
            {
                $writer->startTag('compileoutput');
		$writer->characters($result->{'compileoutput'});
		$writer->endTag('compileoutput');
            }
		$writer->endTag('operand2');

            }
		$writer->endTag('operand1');
        }
	$writer->endTag('binary');
    }
	$writer->endTag('operator');
}

$writer->endTag("operators");

$writer->end();

